#include "testutils.h"
#include "knuth-lfib.h"
#include "nettle-internal.h"

#define SALT "This is a magic salt"
#define MSG1 "None so blind as those who will not see"
#define MSG2 "Fortune knocks once at every man's door"

typedef int (*test_pss_sign_tr_func) (const struct rsa_public_key *pub,
				      const struct rsa_private_key *key,
				      void *random_ctx, nettle_random_func *random,
				      size_t salt_length, const uint8_t *salt,
				      const uint8_t *digest,
				      mpz_t s);

typedef int (*test_pss_verify_func) (const struct rsa_public_key *key,
				     size_t salt_length,
				     const uint8_t *digest,
				     const mpz_t signature);

static void
test_rsa_pss_sign_tr(struct rsa_public_key *pub,
		     struct rsa_private_key *key,
		     test_pss_sign_tr_func sign_tr_func,
		     test_pss_verify_func verify_func,
		     void *ctx, const struct nettle_hash *hash,
		     size_t salt_length, const uint8_t *salt,
		     size_t length, const uint8_t *message,
		     mpz_t expected)
{
  mpz_t signature;
  struct knuth_lfib_ctx lfib;
  uint8_t digest[NETTLE_MAX_HASH_DIGEST_SIZE];
  uint8_t bad_digest[NETTLE_MAX_HASH_DIGEST_SIZE];

  knuth_lfib_init(&lfib, 1111);

  hash->init(ctx);
  hash->update(ctx, length, message);
  hash->digest(ctx, digest);

  mpz_init(signature);

  mpz_set_ui (signature, 17);
  /* Try bad private key */
  mpz_add_ui(key->p, key->p, 2);

  ASSERT(!sign_tr_func(pub, key,
		       &lfib, (nettle_random_func *) knuth_lfib_random,
		       salt_length, salt,
		       digest, signature));

  mpz_sub_ui(key->p, key->p, 2);

  ASSERT(!mpz_cmp_ui(signature, 17));

  /* Try the good private key */
  ASSERT(sign_tr_func(pub, key,
		      &lfib, (nettle_random_func *) knuth_lfib_random,
		      salt_length, salt,
		      digest, signature));

  if (verbose)
    {
      fprintf(stderr, "rsa-pss-tr signature: ");
      mpz_out_str(stderr, 16, signature);
      fprintf(stderr, "\nrsa-pss-tr expected: ");
      mpz_out_str(stderr, 16, expected);
      fprintf(stderr, "\n");
    }

  ASSERT (mpz_cmp(signature, expected) == 0);

  /* Try bad digest */
  memset(bad_digest, 0x17, sizeof(bad_digest));
  ASSERT (!verify_func(pub, salt_length, bad_digest, signature));

  /* Try the good digest */
  ASSERT (verify_func(pub, salt_length, digest, signature));

  /* Try bad signature */
  mpz_combit(signature, 17);
  ASSERT (!verify_func(pub, salt_length, digest, signature));

  mpz_clear(signature);
}


void
test_main(void)
{
  struct rsa_public_key pub;
  struct rsa_private_key key;
  struct sha256_ctx sha256ctx;
  struct sha384_ctx sha384ctx;
  struct sha512_ctx sha512ctx;
  mpz_t p1;
  mpz_t q1;
  struct tstring *salt;
  struct tstring *msg;

  mpz_t expected;

  mpz_init(expected);

  mpz_init(p1);
  mpz_init(q1);

  rsa_private_key_init(&key);
  rsa_public_key_init(&pub);

  test_rsa_set_key_1(&pub, &key);

  /* Test signatures */
  mpz_set_str(expected,
	      "25e6ce0cc00e917e177a09cb4dfd843d104c179b71aded60e68ebc"
	      "ca2cabb1e51502adf28e53fa7ede42619f21a1162755b9658edf88"
	      "a038bb4fea2bb73306fb384d5785c1a8c98a255277c91a4f88ddd3"
	      "52ebdc78f71f7e62b7a870dac4ab25f1004453457e831a1572f7c9"
	      "23fcc48e3b69db582127d14471c7195dce", 16);

  test_rsa_pss_sign_tr(&pub, &key,
		       rsa_pss_sha256_sign_digest_tr,
		       rsa_pss_sha256_verify_digest,
		       &sha256ctx, &nettle_sha256,
		       LDATA(SALT), LDATA(MSG1), expected);

  mpz_set_str(expected,
	      "52f4393ccc92b5672dd3cfd8624765d3a4cdb50c7a92060c33b4663"
	      "fa545b32ce56ec8cd44fe9720df301906ae40921e844b6d80331194"
	      "972f98e309c937c887c53da940778f29d52dd9489e6016a07e9aa16"
	      "b1ea8fefc0860ad69068ad6f94a4b0c8fc8a0797b08c58cf4a8df90"
	      "ee1375feedf7bf73f16ebb2d1cc7e4", 16);

  test_rsa_pss_sign_tr(&pub, &key,
		       rsa_pss_sha256_sign_digest_tr,
		       rsa_pss_sha256_verify_digest,
		       &sha256ctx, &nettle_sha256,
		       LDATA(SALT), LDATA(MSG2), expected);

  /* 777-bit key, generated by
   *
   *   lsh-keygen -a rsa -l 777 -f advanced-hex
   *
   * Interesting because the size of n doesn't equal the sum of the
   * sizes of p and q.
   *
   * (private-key (rsa-pkcs1
   *        (n #013b04440e3eef25 d51c738d508a7fa8 b3445180c342af0f
   *            4cb5a789047300e2 cfc5c5450974cfc2 448aeaaa7f43c374
   *            c9a3b038b181f2d1 0f1a2327fd2c087b a49bf1086969fd2c
   *            d1df3fd69f81fa4b 162cc8bbb363fc95 b7b24b9c53d0c67e
   *            f52b#)
   *        (e #3f1a012d#)
   *        (d #f9bae89dacca6cca c21e0412b4df8355 6fe7c5322bbae8ad
   *            3f11494fd12bc076 d4a7da3050fe109d 2074db09cc6a93b4
   *            745479522558379e a0ddfa74f86c9e9e a22c3b0e93d51447
   *            0feb38105dd35395 63b91ee32776f40c 67b2a175690f7abb
   *            25#)
   *        (p #0b73c990eeda0a2a 2c26416052c85560 0c5c0f5ce86a8326
   *            166acea91786237a 7ff884e66dbfdd3a ab9d9801414c1506
   *            8b#)
   *        (q #1b81c19a62802a41 9c99283331b0badb 08eb0c25ffce0fbf
   *            50017850036f32f3 2132a845b91a5236 61f7b451d587383f
   *            e1#)
   *        (a #0a912fc93a6cca6b 3521725a3065b3be 3c9745e29c93303d
   *            7d29316c6cafa4a2 89945f964fcdea59 1f9d248b0b6734be
   *            c9#)
   *        (b #1658eca933251813 1eb19c77aba13d73 e0b8f4ce986d7615
   *            764c6b0b03c18146 46b7f332c43e05c5 351e09006979ca5b
   *            05#)
   *        (c #0114720dace7b27f 2bf2850c1804869f 79a0aad0ec02e6b4
   *            05e1831619db2f10 bb9b6a8fd5c95df2 eb78f303ea0c0cc8
   *            06#)))
   */

  mpz_set_str(pub.n,
	      "013b04440e3eef25" "d51c738d508a7fa8" "b3445180c342af0f"
	      "4cb5a789047300e2" "cfc5c5450974cfc2" "448aeaaa7f43c374"
	      "c9a3b038b181f2d1" "0f1a2327fd2c087b" "a49bf1086969fd2c"
	      "d1df3fd69f81fa4b" "162cc8bbb363fc95" "b7b24b9c53d0c67e"
	      "f52b", 16);

  mpz_set_str(pub.e, "3f1a012d", 16);

  ASSERT (rsa_public_key_prepare(&pub));

  mpz_set_str(key.p,
	      "0b73c990eeda0a2a" "2c26416052c85560" "0c5c0f5ce86a8326"
	      "166acea91786237a" "7ff884e66dbfdd3a" "ab9d9801414c1506"
	      "8b", 16);

  mpz_set_str(key.q,
	      "1b81c19a62802a41" "9c99283331b0badb" "08eb0c25ffce0fbf"
	      "50017850036f32f3" "2132a845b91a5236" "61f7b451d587383f"
	      "e1", 16);

  mpz_set_str(key.a,
	      "0a912fc93a6cca6b" "3521725a3065b3be" "3c9745e29c93303d"
	      "7d29316c6cafa4a2" "89945f964fcdea59" "1f9d248b0b6734be"
	      "c9", 16);

  mpz_set_str(key.b,
	      "1658eca933251813" "1eb19c77aba13d73" "e0b8f4ce986d7615"
	      "764c6b0b03c18146" "46b7f332c43e05c5" "351e09006979ca5b"
	      "05", 16);

  mpz_set_str(key.c,
	      "0114720dace7b27f" "2bf2850c1804869f" "79a0aad0ec02e6b4"
	      "05e1831619db2f10" "bb9b6a8fd5c95df2" "eb78f303ea0c0cc8"
	      "06", 16);

  ASSERT (rsa_private_key_prepare(&key));
  ASSERT (pub.size == key.size);

  /* Test signatures */
  mpz_set_str(expected,
	      "1a4d28331341cabf7ac85bc59a58d439b7ec2c607c6a74e35b5909"
	      "1dfa3d9de9fde93e4a431f0f768bec07c39995d253209f86e3dc84"
	      "037ecd5d23d963fab4fa8a001e018d82cb19d743a94ba7dc7a821e"
	      "87b72e67a0fe058f956208f7060dc104", 16);

  test_rsa_pss_sign_tr(&pub, &key,
		       rsa_pss_sha256_sign_digest_tr,
		       rsa_pss_sha256_verify_digest,
		       &sha256ctx, &nettle_sha256,
		       LDATA(SALT), LDATA(MSG1), expected);

  /* From FIPS 186-2 */
  mpz_set_str(pub.n,
	      "be499b5e7f06c83f" "a0293e31465c8eb6" "b58af920bae52a7b"
	      "5b9bfeb7aa72db12" "64112eb3fd431d31" "a2a7e50941566929"
	      "494a0e891ed56139" "18b4b51b0d1fb977" "83b26acf7d0f384c"
	      "fb35f4d2824f5dd3" "80623a26bf180b63" "961c619dcdb20cae"
	      "406f22f6e276c80a" "37259490cfeb72c1" "a71a84f1846d3308"
	      "77ba3e3101ec9c7b" , 16);

  mpz_set_str(pub.e, "11", 16);

  ASSERT (rsa_public_key_prepare(&pub));

  mpz_set_str(key.p,
	      "e7a80c5d211c06ac" "b900939495f26d36" "5fc2b4825b75e356"
	      "f89003eaa5931e6b" "e5c3f7e6a633ad59" "db6289d06c354c23"
	      "5e739a1e3f3d39fb" "40d1ffb9cb44288f", 16);

  mpz_set_str(key.q,
	      "d248aa248000f720" "258742da67b71194" "0c8f76e1ecd52b67"
	      "a6ffe1e49354d66f" "f84fa601804743f5" "838da2ed4693a5a2"
	      "8658d6528cc1803b" "f6c8dc73c5230b55", 16);

  mpz_set_str(key.d,
	      "0d0f17362bdad181" "db4e1fe03e8de1a3" "208989914e14bf26"
	      "9558826bfa20faf4" "b68dba6bb989a01f" "03a21c44665dc5f6"
	      "48cb5b59b954eb10" "77a80263bd22cdfb" "88d39164b7404f4f"
	      "1106ee01cf60b776" "95748d8fdaf9fd42" "8963fe75144010b1"
	      "934c8e26a8823967" "2cf49b3422a07c4d" "834ba208d570fe40"
	      "8e7095c90547e68d", 16);

  /* a = d % (p-1) */
  mpz_sub_ui(p1, key.p, 1);
  mpz_fdiv_r(key.a, key.d, p1);
  mpz_clear(p1);

  /* b = d % (q-1) */
  mpz_sub_ui(q1, key.q, 1);
  mpz_fdiv_r(key.b, key.d, q1);
  mpz_clear(q1);

  /* c = q^{-1} (mod p) */
  mpz_invert(key.c, key.q, key.p);

  ASSERT (rsa_private_key_prepare(&key));
  ASSERT (pub.size == key.size);

  mpz_set_str(expected,
	      "11e169f2fd40b07641b9768a2ab19965fb6c27f10fcf0323fcc6d1"
	      "2eb4f1c06b330ddaa1ea504407afa29de9ebe0374fe9d1e7d0ffbd"
	      "5fc1cf3a3446e4145415d2ab24f789b3464c5c43a256bbc1d692cf"
	      "7f04801dac5bb401a4a03ab7d5728a860c19e1a4dc797ca542c820"
	      "3cec2e601eb0c51f567f2eda022b0b9ebddeeefa", 16);

  salt = SHEX("11223344555432167890");
  msg = SHEX("c7f5270fca725f9bd19f519a8d7cca3cc5c079024029f3bae510f9"
	     "b02140fe238908e4f6c18f07a89c687c8684669b1f1db2baf9251a"
	     "3c829faccb493084e16ec9e28d58868074a5d6221667dd6e528d16"
	     "fe2c9f3db4cfaf6c4dce8c8439af38ceaaaa9ce2ecae7bc8f4a5a5"
	     "5e3bf96df9cd575c4f9cb327951b8cdfe4087168");

  test_rsa_pss_sign_tr(&pub, &key,
		       rsa_pss_sha256_sign_digest_tr,
		       rsa_pss_sha256_verify_digest,
		       &sha256ctx, &nettle_sha256,
		       salt->length, salt->data, msg->length, msg->data,
		       expected);

  mpz_set_str(expected,
	      "b281ad934b2775c0cba5fb10aa574d2ed85c7f99b942b78e497024"
	      "80069362ed394baded55e56cfcbe7b0b8d2217a05a60e1acd725cb"
	      "09060dfac585bc2132b99b41cdbd530c69d17cdbc84bc6b9830fc7"
	      "dc8e1b2412cfe06dcf8c1a0cc3453f93f25ebf10cb0c90334fac57"
	      "3f449138616e1a194c67f44efac34cc07a526267", 16);

  test_rsa_pss_sign_tr(&pub, &key,
		       rsa_pss_sha384_sign_digest_tr,
		       rsa_pss_sha384_verify_digest,
		       &sha384ctx, &nettle_sha384,
		       salt->length, salt->data, msg->length, msg->data,
		       expected);
  mpz_set_str(expected,
	      "8ffc38f9b820ef6b080fd2ec7de5626c658d79056f3edf610a295b"
	      "7b0546f73e01ffdf4d0070ebf79c33fd86c2d608be9438b3d420d0"
	      "9535b97cd3d846ecaf8f6551cdf93197e9f8fb048044473ab41a80"
	      "1e9f7fc983c62b324361dade9f71a65952bd35c59faaa4d6ff462f"
	      "68a6c4ec0b428aa47336f2178aeb276136563b7d", 16);

  test_rsa_pss_sign_tr(&pub, &key,
		       rsa_pss_sha512_sign_digest_tr,
		       rsa_pss_sha512_verify_digest,
		       &sha512ctx, &nettle_sha512,
		       salt->length, salt->data, msg->length, msg->data,
		       expected);

  /* The public key n for this test is of size k = 1017 bits, and the
     pss "em" value is limited to k - 1 = 1016 bits or 127 octets. The
     alleged signature below results in a 1017 bit number during the
     signature verification, which is too large, and used to result in
     an assertion failure when attempting to convert the number to a
     127 octet string.
  */
  mpz_set_str(pub.n,
	      "1d64559685aad3490e976b48aacf442ecee847268f882341eafe78"
	      "a0ca4ef88f66edbaf55b70e5285cc117aa9ceb322a4227c17e9e89"
	      "27bf38e5672faecf79e2983d92766fbb6624522f072ae0e4e46d37"
	      "052ce1e5745c2dd8fd67de3862e4711161e359b96bda85911ebf4e"
	      "6ce1bea625970269c77004a3cb03f9c382c5f79", 16);
  mpz_set_str(pub.e, "10001", 16);

  ASSERT (rsa_public_key_prepare(&pub));

  msg = SHEX("7f85e4909ff7bb29536e540a53031ef03ddcb129e553a43273fa1f"
	     "ed28c22a8b57c7bde101ff746f335ba69b29642019");
  /* Alleged signature, resulting in a too large m. */
  mpz_set_str(expected,
	      "000000000000000000000000000000000000000000000000000000"
	      "000000000000000000000000000000000000000000000005ffff05"
	      "000000000000000000000000000000000000000000000000000000"
	      "000000000000000000000000000000000000000000000000000000"
	      "0000000000000000000000000000000000000000", 16);

  ASSERT(!rsa_pss_sha384_verify_digest(&pub, 48, msg->data, expected));

  rsa_private_key_clear(&key);
  rsa_public_key_clear(&pub);
  mpz_clear(expected);
}
